#include <Nordic\reg24le1.h>
#include <stdint.h>
#include "hal_spi.h"
#include "lib_at25df021.h"
#include "intrins.h"

#define AT25_CONFIG_DATA_ADDRESS 0x0400
#define AT25_CONFIG_DATA_BLOCK   0

void generate_random_bytes(uint8_t *data_array, uint8_t array_length)
{
  // Enabled the random generator with bias correction set
  RNGCTL = 0xC0;
  while(array_length--)
  {
    // Wait for the data ready bit to be set
    while((RNGCTL & 0x20) == 0);
    // Write the random data to the array
    *data_array++ = RNGDAT;
  }
}

void main()
{
  xdata uint8_t serial_number[16];

  // Set P00 to output ( SPI chip select )
  P0DIR &= ~0x01;
  // SPI chip select should be high when the SPI interface is inactive
  SPI_CHIP_SELECT = 1;

  // Configure master SPI interface for compatibility with the AT25
  hal_spi_master_init(SPI_CLK_DIV2, HAL_SPI_MODE_0, HAL_SPI_MSB_LSB);

  // If the first byte of the designated serial number area is not 0xA3 the area is considered empty and the serial code must be written to the flash
  if( at25_read_byte(AT25_CONFIG_DATA_BLOCK, AT25_CONFIG_DATA_ADDRESS) != 0xA3 )
  {
    // Generate a random serial number
    generate_random_bytes(serial_number, 16);
    // By default all sensors in the flash will be protected, and to allow write access the unprotect command must be run
    at25_unprotect_sector(AT25_CONFIG_DATA_BLOCK);
    // The block is erased to make sure all bytes are 0xFF
    at25_block_erase_64kb(AT25_CONFIG_DATA_BLOCK);
    // The first byte is set to 0xA3 to inform the system that the serial number is written into flash
    at25_program_byte(AT25_CONFIG_DATA_BLOCK, AT25_CONFIG_DATA_ADDRESS, 0xA3);
    // Write the serial number, starting at the second byte of the designated flash area
    at25_program_bytes(AT25_CONFIG_DATA_BLOCK, AT25_CONFIG_DATA_ADDRESS + 1, serial_number, 16);
  }
  else
  {
    // If the serial number is already written to the flash we read it back out
    at25_read_bytes(AT25_CONFIG_DATA_BLOCK, AT25_CONFIG_DATA_ADDRESS + 1, serial_number, 16);
  }

  // The serial number can be found in the serial_number array. The first time the program runs it is randomly generated, and all following runs it will be read from the flash.   
  while(1);
}